/*****************************************************************************
 $Id: StegoAlgoLSBpp.java,v 1.7 2002/04/14 03:42:33 bastian Exp $
 Part of TRex, (c) 2001, 2002 Bastian Friedrich <bastian@bastian-friedrich.de>
 This software licensed under the GPL.
 *****************************************************************************/
package trex.Algo;

import trex.Filter.*;
import trex.*;

/**
 * This class adds a (very little) bit of security to the LSB algorithm by
 * XORing every input character with a character from the passphrase.
 * @author Bastian Friedrich <a href="mailto:bastian@bastian-friedrich.de">&lt;bastian@bastian-friedrich.de&gt;</a>
 * @version $Revision: 1.7 $
 */

public class StegoAlgoLSBpp extends StegoAlgoLSB {

  /**
   * This class adds the XOR functionality to the "getData" function.
   * Besides that, it is identical to the LSBDecryptFilter.
   */
  protected class LSBppDecryptFilter extends LSBDecryptFilter {

    /**
     * Constructor. Calls <code>super()</code>.
     */
    public LSBppDecryptFilter(int w, int h) {
      super(w, h);
    }

    /**
     * Return contained data as a {@link java.lang.String}.
     * @return Contained data.
     */
    public String getData() throws DecryptImpossibleException {
      if (data == null) return null;
      int datalen = TRexUtil.bytesToInt(TRexUtil.subArray(data, 0, 4));

      if ((datalen < 0) || (datalen > data.length-4))
        throw new DecryptImpossibleException();

      byte xordata[] = TRexUtil.subArray(data, 4, datalen);
      byte oridata[] = new byte[datalen];
      byte passPhraseData[];
      int curPos;

      if (passPhrase == null)
        oridata = xordata;        /* Pass phrase set at all? */
      else {
        passPhraseData = passPhrase.getBytes();
        for (int i = 0; i < datalen; i++) {
          curPos = i % passPhrase.length(); /* Get position in pass phrase to XOR with */
          oridata[i] = (byte)(xordata[i] ^ passPhraseData[curPos]); /* calculate output data */
        }
      }

      return new String(oridata);
    }

  }

  /**
   * Return true.
   */
  public boolean hasPassPhrase() {
    return true;
  }

  /**
   * Return an info string about this algorithm.
   * Contains the algorithm's name.
   * @return "Least significant bit algorithm, XORing pass phrase".
   */
  public String getInfo() {
    return "Least significant bit algorithm, XORing pass phrase";
  }

  /**
   * Return an encrypt filter for this algorithm (identical to LSB)
   * and XOR data with pass phrase beforehand. Actually called by "virtual"
   * method {@link trex.Algo.StegoAlgoRGB#getDecrypted}.
   * @param data Data to hide while filtering
   * @param w Picture's width
   * @param h Picture's height
   * @return The {@link java.awt.image.ImageFilter} that does all the work.
   */
  protected TRexFilterRGB getEncryptFilter(String data, int w, int h) {
    byte xordata[] = new byte[data.length()];
    byte oridata[] = data.getBytes();
    byte passPhraseData[];
    int curPos;
    /* see LSBppDecryptFilter above... (Well, this is in fact trivial) */
    if ((data == null)
     || (data.equals(""))
     || (passPhrase == null)
     || (passPhrase.equals("")))
      xordata = oridata;
    else {
      passPhraseData = passPhrase.getBytes();
      for (int i = 0; i < data.length(); i++) {
        curPos = i % passPhrase.length();
        xordata[i] = (byte)(oridata[i] ^ passPhraseData[curPos]);
      }
    }
    return new LSBEncryptFilter(new String(xordata), w, h);
  }

  /**
   * Return a decrypt filter for this algorithm. Actually called by "virtual"
   * method {@link trex.Algo.StegoAlgoRGB#getDecrypted}.
   * @param w Picture's width
   * @param h Picture's height
   * @return The {@link java.awt.image.ImageFilter} that does all the work.
   */
  public TRexFilterRGB getDecryptFilter(int w, int h) {
    return new LSBppDecryptFilter(w, h);
  }

}